import React, {
    forwardRef,
    useImperativeHandle,
    useRef,
    useState,
} from 'react';

import {
    MapPinPen,
    XIcon,
} from 'lucide-react';

import { GeoRegionGType } from '@/api/types';
import { Button } from '@/components/ui/button';
import {
    Dialog,
    DialogContent,
    DialogFooter,
} from '@/components/ui/dialog';
import { useAppConfig } from '@/contexts/AppConfigContext';
import { Lang } from '@/lang/Lang';

import GmMap
    from './CollectionFields/components/ItemData/GeoRegionData/components/GmMap';
import {
    MobileFab,
    MobileLink,
    MobileNavbar,
    MobilePage,
    MobilePopup,
} from './ui/mobile';

type OpenParams = {
    initialValues: GeoRegionGType[];
    onSave: (values: GeoRegionGType[]) => void;
}
type DialogRegionEditorRef = {
    open: (params: OpenParams) => void;
    close: () => void;
}

export const dialogRegionEditorRef = React.createRef<DialogRegionEditorRef>();

const DialogRegionEditor = forwardRef<DialogRegionEditorRef, {}>((_, ref) => {
    const [open, setOpen] = useState(false);

    const [values, setValues] = useState<GeoRegionGType[]>([]);

    const onSaveRef = useRef<((values: GeoRegionGType[]) => void) | null>(null);

    const { isMobile } = useAppConfig();

    useImperativeHandle(ref, () => ({
        open: (params: OpenParams) => {
            setValues(params.initialValues);
            onSaveRef.current = params.onSave;
            setOpen(true);
        },
        close: () => {
            setOpen(false);
        }
    }));

    if (isMobile) {

        return <MobilePopup
            opened={open}
            onBackdropClick={() => setOpen(false)}
        >
            <MobilePage className="overflow-hidden">
                <MobileNavbar
                    title={Lang.key('region_editor')}
                    right={<MobileLink onClick={() => setOpen(false)}><XIcon /></MobileLink>}
                />
                {open && <GmMap
                    height={window.innerHeight}
                    onlyView={false}
                    values={values}
                    onChange={(val) => {
                        setValues(val);
                    }}
                />}

                <MobileFab
                    className='fixed left-1/2 bottom-safe-4 transform -translate-x-1/2 z-20'
                    icon={<MapPinPen />}
                    onClick={() => {
                        onSaveRef.current?.(values);
                        setOpen(false);
                    }}
                    text={Lang.key("save")}
                />


            </MobilePage>


        </MobilePopup>

    }

    return <Dialog
        open={open}
        onOpenChange={setOpen}
    >
        <DialogContent className="!w-[80vw] !max-w-[80vw] sm:!max-w-[80vw] overflow-hidden ">
            {open && <GmMap
                height={window.innerHeight - 200}
                onlyView={false}
                values={values}
                onChange={(val) => {
                    setValues(val);
                }}
            />}
            <DialogFooter>
                <Button onClick={() => {
                    setOpen(false);
                }}>
                    {Lang.key('cancel')}
                </Button>
                <Button onClick={() => {
                    onSaveRef.current?.(values);
                    setOpen(false);
                }}>
                    {Lang.key('save')}
                </Button>
            </DialogFooter>
        </DialogContent>

    </Dialog>

});

export default DialogRegionEditor;