import {
    createRef,
    forwardRef,
    useEffect,
    useImperativeHandle,
    useState,
} from 'react';

import {
    toast as toastSonner,
    Toaster,
} from 'sonner';

import { useAppConfig } from '@/contexts/AppConfigContext';

import { MobileNotification } from './ui/mobile';

type ToastRootType = {
    success: (message: string | React.ReactNode, data?: any) => string | number;
    info: (message: string | React.ReactNode, data?: any) => string | number;
    warning: (message: string | React.ReactNode, data?: any) => string | number;
    error: (message: string | React.ReactNode, data?: any) => string | number;

    dismiss: (id?: number | string) => string | number;
    loading: (message: string | React.ReactNode, data?: any) => string | number;
}

export const toast: ToastRootType = {
    success: (message, data) => {
        if (toastRootRef.current) {
            return toastRootRef.current?.success(message, data);
        } else {
            return toastSonner.success(message, data);
        }
    },
    info: (message, data) => {
        if (toastRootRef.current) {
            return toastRootRef.current?.info(message, data);
        } else {
            return toastSonner.info(message, data);
        }
    },
    warning: (message, data) => {
        if (toastRootRef.current) {
            return toastRootRef.current?.warning(message, data);
        } else {
            return toastSonner.warning(message, data);
        }
    },
    error: (message, data) => {
        if (toastRootRef.current) {
            return toastRootRef.current?.error(message, data);
        } else {
            return toastSonner.error(message, data);
        }
    },
    dismiss: (id) => {
        if (toastRootRef.current) {
            return toastRootRef.current?.dismiss(id);
        } else {
            return toastSonner.dismiss(id);
        }
    },
    loading: (message, data) => {
        if (toastRootRef.current) {
            return toastRootRef.current?.loading(message, data);
        } else {
            return toastSonner.loading(message, data);
        }
    },
}


type ToastRootRefType = {
    success: (message: string | React.ReactNode, data?: any) => string | number;
    info: (message: string | React.ReactNode, data?: any) => string | number;
    warning: (message: string | React.ReactNode, data?: any) => string | number;
    error: (message: string | React.ReactNode, data?: any) => string | number;
    dismiss: (id?: number | string) => string | number;
    loading: (message: string | React.ReactNode, data?: any) => string | number;
}

type MobileToastType = {
    message: string | React.ReactNode;
    type: "success" | "info" | "warning" | "error" | "loading";
}

export const toastRootRef = createRef<ToastRootRefType>();

const ToastRoot = forwardRef<ToastRootRefType, {}>((_, ref) => {

    const [mobileToast, setMobileToast] = useState<MobileToastType | null>(null);

    const { isMobile, title, icon } = useAppConfig();

    useImperativeHandle(ref, () => ({
        success: (message, data) => {
            if (isMobile) {
                setMobileToast({ message, type: "success" });
                return 0;
            } else {
                return toastSonner.success(message, data);
            }
        },
        info: (message, data) => {
            if (isMobile) {
                setMobileToast({ message, type: "info" });
                return 0;
            } else {
                return toastSonner.info(message, data);
            }
        },
        warning: (message, data) => {
            if (isMobile) {
                setMobileToast({ message, type: "warning" });
                return 0;
            } else {
                return toastSonner.warning(message, data);
            }
        },
        error: (message, data) => {
            if (isMobile) {
                setMobileToast({ message, type: "error" });
                return 0;
            } else {
                return toastSonner.error(message, data);
            }
        },
        dismiss: (id) => {
            if (isMobile) {
                setMobileToast(null);
                return 0;
            } else {
                return toastSonner.dismiss(id);
            }
        },
        loading: (message, data) => {
            if (isMobile) {
                setMobileToast({ message, type: "loading" });
                return 0;
            } else {
                return toastSonner.loading(message, data);
            }
        },
    }));

    useEffect(() => {
        if (mobileToast && mobileToast.type !== "loading") {
            const timer = setTimeout(() => {
                setMobileToast(null);
            }, 3000);
            return () => clearTimeout(timer);
        }
    }, [mobileToast, isMobile]);



    if (isMobile) {
        return <MobileNotification
            opened={!!mobileToast}
            icon={<img src={icon} className='w-6 h-6 rounded-sm' />}
            onClick={() => {
                setMobileToast(null);
            }}
            title={title}
            button
            // titleRightText="now"
            subtitle={mobileToast?.message}
        // text={mobileToast?.message}
        />
    }

    return <Toaster
        position="top-right"
    />
});

export default ToastRoot;