import fs from 'fs';
import path from 'path';
import {
  defineConfig,
  type Plugin,
} from 'vite';

import tailwindcss from '@tailwindcss/vite';
import react from '@vitejs/plugin-react';

// Плагин для автоматического добавления отсутствующих ключей в local.ts
function addLangKeyPlugin(): Plugin {
  return {
    name: 'add-lang-key',
    configureServer(server) {
      server.middlewares.use('/api/add-lang-key', async (req, res, next) => {
        if (req.method === 'POST') {
          let body = '';
          req.on('data', chunk => {
            body += chunk.toString();
          });
          req.on('end', async () => {
            try {
              const { key } = JSON.parse(body);
              const localFilePath = path.resolve(__dirname, './src/lang/local.ts');

              // Читаем текущий файл
              let content = fs.readFileSync(localFilePath, 'utf-8');

              // Проверяем, есть ли уже такой ключ
              if (content.includes(`'${key}':`)) {
                res.writeHead(200, { 'Content-Type': 'application/json' });
                res.end(JSON.stringify({ success: true, message: 'Ключ уже существует' }));
                return;
              }

              // Находим место для вставки (перед закрывающей скобкой объекта local)
              // Ищем последнюю закрывающую скобку объекта local (не типа)
              const lines = content.split('\n');
              let insertLineIndex = -1;
              for (let i = lines.length - 1; i >= 0; i--) {
                if (lines[i].trim() === '}') {
                  insertLineIndex = i;
                  break;
                }
              }

              if (insertLineIndex === -1) {
                throw new Error('Не найдена закрывающая скобка объекта local');
              }

              // Формируем новый ключ
              const newKey = `    '${key}': {\n        ru: "НЕ Задано",\n        en: "НЕ Задано",\n    },\n`;

              // Проверяем, нужна ли запятая после последнего ключа
              let needsComma = false;
              for (let i = insertLineIndex - 1; i >= 0; i--) {
                const line = lines[i].trim();
                if (line && !line.startsWith('//')) {
                  needsComma = !line.endsWith(',');
                  break;
                }
              }

              // Вставляем новый ключ перед закрывающей скобкой
              const beforeLines = lines.slice(0, insertLineIndex);
              const afterLines = lines.slice(insertLineIndex);

              let newContent = beforeLines.join('\n');
              if (needsComma && newContent.trim().length > 0) {
                // Добавляем запятую к последней непустой строке
                for (let i = beforeLines.length - 1; i >= 0; i--) {
                  if (beforeLines[i].trim() && !beforeLines[i].trim().startsWith('//')) {
                    if (!beforeLines[i].trim().endsWith(',')) {
                      beforeLines[i] = beforeLines[i].trimEnd() + ',';
                    }
                    break;
                  }
                }
                newContent = beforeLines.join('\n');
              }

              newContent += '\n' + newKey + afterLines.join('\n');

              // Записываем обновленный файл
              fs.writeFileSync(localFilePath, newContent, 'utf-8');

              res.writeHead(200, { 'Content-Type': 'application/json' });
              res.end(JSON.stringify({ success: true, message: `Ключ "${key}" добавлен` }));
            } catch (error) {
              res.writeHead(500, { 'Content-Type': 'application/json' });
              res.end(JSON.stringify({ success: false, error: String(error) }));
            }
          });
        } else {
          next();
        }
      });
    },
  };
}



// https://vite.dev/config/
export default defineConfig({
  plugins: [react(), tailwindcss(), addLangKeyPlugin()],
  resolve: {
    alias: {
      "@": path.resolve(__dirname, "./src"),
    },
  },
  build: {
    rollupOptions: {
      output: {
        entryFileNames: 'main.js',
        assetFileNames: (assetInfo) => {
          if (assetInfo.name && assetInfo.name.endsWith('.css')) {
            return 'main.css';
          }
          return assetInfo.name || 'asset';
        },
      },
    },
  },
})